<?php

// Set execution time to 5 minutes max to avoid timeouts, but respect shared hosting limits
ini_set('max_execution_time', 300);
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../lib/Config.php';
require_once __DIR__ . '/../lib/DB.php';
require_once __DIR__ . '/../lib/Scraper.php';
require_once __DIR__ . '/../lib/Rewriter.php';
require_once __DIR__ . '/../lib/Indexer.php';
require_once __DIR__ . '/../lib/Logger.php';

// Load Config
Config::load(__DIR__ . '/../.env');

Logger::info("Autopost Cron Started");

// 1. Get Trends
$trends = Scraper::getGoogleTrends('US');
Logger::info("Fetched " . count($trends) . " trends");

$postsCreated = 0;
$maxPostsPerRun = 20; // Increased limit for higher volume

foreach ($trends as $trend) {
    if ($postsCreated >= $maxPostsPerRun) break;

    $title = $trend['title'];
    
    // 2. Deduplication Check
    // Check if we have a post with a similar title recently
    $existing = DB::fetch("SELECT id FROM posts WHERE title LIKE ? LIMIT 1", ["%$title%"]);
    if ($existing) {
        Logger::info("Skipping duplicate: $title");
        continue;
    }

    // 3. Fetch Source Content
    $newsUrl = $trend['news_item_url'];
    if (!$newsUrl) continue;

    Logger::info("Processing trend: $title - Source: $newsUrl");
    
    $html = Scraper::fetchUrl($newsUrl);
    $rawContent = Scraper::extractArticleContent($html);

    if (strlen($rawContent) < 500) {
        Logger::warning("Content too short for $title, skipping.");
        continue;
    }

    // 4. Rewrite Content
    $rewrittenContent = Rewriter::rewrite($rawContent, "Write a respectful and detailed obituary or biography for '$title' based on the following text. Focus on their life, achievements, and legacy. Use a neutral, journalistic tone.");
    
    // Generate a slug
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
    $slug .= '-' . date('Ymd');

    // 5. Save Post
    try {
        $postId = DB::insert('posts', [
            'title' => $title,
            'slug' => $slug,
            'content' => $rewrittenContent,
            'source_url' => $newsUrl,
            'source_name' => 'Google Trends',
            'fingerprint' => md5($title), // Simple fingerprint for now
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s')
        ]);

        Logger::info("Post created: $title (ID: $postId)");
        $postsCreated++;

        // 6. Indexing
        $postUrl = Config::get('APP_URL') . '/post.php?slug=' . $slug;
        Indexer::requestIndexing($postUrl);

    } catch (Exception $e) {
        Logger::error("Failed to save post $title: " . $e->getMessage());
    }
}

// Update Stats
$today = date('Y-m-d');
DB::query("INSERT INTO stats_daily (day, posts_published) VALUES (?, ?) ON DUPLICATE KEY UPDATE posts_published = posts_published + ?", [$today, $postsCreated, $postsCreated]);

Logger::info("Autopost Cron Finished. Created $postsCreated posts.");
