<?php
require_once __DIR__ . '/../../lib/Config.php';
require_once __DIR__ . '/../../lib/DB.php';
require_once __DIR__ . '/../../lib/Auth.php';
require_once __DIR__ . '/../../lib/Analytics.php';

Config::load(__DIR__ . '/../../.env');
Auth::requireAdmin();

// Fetch Stats
$stats = [
    'posts' => DB::fetch("SELECT COUNT(*) as c FROM posts")['c'],
    'views_total' => DB::fetch("SELECT SUM(views) as c FROM posts")['c'] ?? 0,
    'views_24h' => Analytics::getStats('24h'),
    'views_7d' => Analytics::getStats('7d'),
    'views_30d' => Analytics::getStats('30d'),
];

// Calculate "Growth" (Fake/Simulated for now, or simple math if we had previous data)
// For demo purposes, we'll just show some static growth percentages like the image
$growth = [
    'posts' => '+12.5%',
    'views' => '+45.3%',
    'engagement' => '+8.4%'
];

$browsers = Analytics::getBrowserStats();
$recentPosts = DB::fetchAll("SELECT * FROM posts ORDER BY created_at DESC LIMIT 10");

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BioVibe Analytics</title>
    <link rel="stylesheet" href="/assets/css/fonts.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <!-- Chart.js -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body style="background: #000;">

    <div class="admin-layout">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-brand">
                BioVibe Analytics
            </div>
            
            <div class="sidebar-menu">
                <div class="sidebar-section-title">Navigation</div>
                <a href="/admin/" class="sidebar-link active">
                    <span>📊</span> Dashboard
                </a>
                <a href="/admin/add-post.php" class="sidebar-link">
                    <span>✍️</span> Add Post
                </a>
                <a href="/admin/settings.php" class="sidebar-link">
                    <span>⚙️</span> Settings
                </a>
                <a href="/admin/logs.php" class="sidebar-link">
                    <span>📝</span> Logs
                </a>
                <a href="/" target="_blank" class="sidebar-link">
                    <span>👀</span> View Site
                </a>
                <a href="/admin/logout.php" class="sidebar-link">
                    <span>🚪</span> Logout
                </a>
            </div>
        </aside>

        <!-- Main Content -->
        <div class="dashboard-content">
            
            <!-- Stats Grid -->
            <div class="stats-grid">
                <!-- Total Posts -->
                <div class="stat-card">
                    <div class="stat-label">
                        <span>Total Posts</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($stats['posts']); ?></div>
                    <div class="stat-trend trend-up">
                        <?php echo $growth['posts']; ?> <span style="color: var(--text-muted); margin-left: 4px;">vs last month</span>
                    </div>
                </div>

                <!-- Total Views -->
                <div class="stat-card">
                    <div class="stat-label">
                        <span>Total Views</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($stats['views_total']); ?></div>
                    <div class="stat-trend trend-up">
                        <?php echo $growth['views']; ?> <span style="color: var(--text-muted); margin-left: 4px;">vs last month</span>
                    </div>
                </div>

                <!-- 24h Views -->
                <div class="stat-card">
                    <div class="stat-label">
                        <span>Views (24h)</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($stats['views_24h']); ?></div>
                    <div class="stat-trend trend-up">
                        <span>Active Now</span>
                    </div>
                </div>

                <!-- 30d Views -->
                <div class="stat-card">
                    <div class="stat-label">
                        <span>Views (30d)</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($stats['views_30d']); ?></div>
                    <div class="stat-trend trend-up">
                        <span>+3.2%</span> <span style="color: var(--text-muted); margin-left: 4px;">vs last month</span>
                    </div>
                </div>
            </div>

            <!-- Charts Section -->
            <div class="dashboard-section">
                <div class="section-header">
                    <h2 class="section-title">Follower Growth</h2>
                    <select style="background: transparent; color: var(--text-muted); border: 1px solid var(--border); padding: 4px 8px; border-radius: 4px;">
                        <option>Last 6 weeks</option>
                    </select>
                </div>
                <div style="height: 300px; width: 100%;">
                    <canvas id="growthChart"></canvas>
                </div>
            </div>

            <!-- Recent Posts & Browsers -->
            <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 2rem;">
                
                <!-- Recent Posts Table -->
                <div class="dashboard-section">
                    <div class="section-header">
                        <h2 class="section-title">Recent Posts</h2>
                        <a href="/admin/add-post.php" class="btn" style="font-size: 0.8rem;">+ New Post</a>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Status</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recentPosts as $post): ?>
                            <tr>
                                <td><?php echo htmlspecialchars(substr($post['title'], 0, 40)) . (strlen($post['title']) > 40 ? '...' : ''); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $post['status']; ?>">
                                        <?php echo ucfirst($post['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('M d', strtotime($post['created_at'])); ?></td>
                                <td>
                                    <a href="/post.php?slug=<?php echo $post['slug']; ?>" target="_blank" style="color: var(--accent);">View</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Browser Stats -->
                <div class="dashboard-section">
                    <div class="section-header">
                        <h2 class="section-title">Top Browsers</h2>
                    </div>
                    <?php foreach ($browsers as $b): ?>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 1rem; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                        <span style="color: var(--text-muted);"><?php echo $b['browser']; ?></span>
                        <span style="color: var(--text-main); font-weight: bold;"><?php echo $b['count']; ?></span>
                    </div>
                    <?php endforeach; ?>
                </div>

            </div>

        </div>
    </div>

    <script>
        // Chart.js Configuration
        const ctx = document.getElementById('growthChart').getContext('2d');
        
        // Create gradient
        const gradient = ctx.createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, 'rgba(0, 242, 234, 0.5)');
        gradient.addColorStop(1, 'rgba(0, 242, 234, 0)');

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: ['Week 1', 'Week 2', 'Week 3', 'Week 4', 'Week 5', 'Week 6'],
                datasets: [{
                    label: 'Views',
                    data: [1200, 1900, 3000, 5000, 4500, 7000],
                    borderColor: '#00f2ea',
                    backgroundColor: gradient,
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4,
                    pointRadius: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: {
                        grid: { color: '#222' },
                        ticks: { color: '#666' }
                    },
                    x: {
                        grid: { display: false },
                        ticks: { color: '#666' }
                    }
                }
            }
        });
    </script>
</body>
</html>
