<?php
require_once __DIR__ . '/../../lib/Config.php';
require_once __DIR__ . '/../../lib/DB.php';
require_once __DIR__ . '/../../lib/Auth.php';

Config::load(__DIR__ . '/../../.env');
Auth::requireAdmin();

$message = null;
$error = null;

// Handle Form Submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // 1. Handle Google Service Account Upload
    if (isset($_FILES['google_json']) && $_FILES['google_json']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../'; // Root directory
        $destPath = $uploadDir . 'google-service-account.json';
        
        $fileType = strtolower(pathinfo($_FILES['google_json']['name'], PATHINFO_EXTENSION));
        if ($fileType !== 'json') {
            $error = "Only JSON files are allowed for Google Service Account.";
        } else {
            if (move_uploaded_file($_FILES['google_json']['tmp_name'], $destPath)) {
                $message = "Google Service Account JSON uploaded successfully.";
            } else {
                $error = "Failed to upload Google Service Account JSON.";
            }
        }
    }

    // 2. Update .env Settings
    if (!$error) {
        $envPath = __DIR__ . '/../../.env';
        $envContent = file_get_contents($envPath);
        
        $updates = [
            'RSS_FEEDS' => $_POST['rss_feeds'] ?? '',
            'AI_API_KEY' => $_POST['ai_api_key'] ?? '',
            'AI_MODEL' => $_POST['ai_model'] ?? '',
            'PLAGIARISM_THRESHOLD' => $_POST['plagiarism_threshold'] ?? ''
        ];

        foreach ($updates as $key => $value) {
            // Simple regex replacement for .env values
            // This assumes keys exist. If not, they won't be added (simplified logic)
            $pattern = "/^$key=.*/m";
            $replacement = "$key=$value";
            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, $replacement, $envContent);
            } else {
                $envContent .= "\n$key=$value";
            }
        }

        if (file_put_contents($envPath, $envContent)) {
            $message = $message ? $message . " & Settings updated." : "Settings updated successfully.";
            // Reload config to reflect changes immediately
            Config::load($envPath); 
        } else {
            $error = "Failed to write to .env file.";
        }
    }
}

// Read current values
$currentRss = Config::get('RSS_FEEDS');
$currentAiKey = Config::get('AI_API_KEY');
$currentAiModel = Config::get('AI_MODEL');
$currentThreshold = Config::get('PLAGIARISM_THRESHOLD');
$googleJsonExists = file_exists(__DIR__ . '/../../google-service-account.json');

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Admin</title>
    <link rel="stylesheet" href="/assets/css/fonts.css">
    <link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
    <header>
        <div class="container">
            <nav>
                <a href="/admin/" class="logo">Admin Panel</a>
                <div class="nav-links">
                    <a href="/admin/">Dashboard</a>
                    <a href="/admin/logout.php">Logout</a>
                </div>
            </nav>
        </div>
    </header>
    
    <main class="container">
        <div class="admin-dashboard">
            <aside class="sidebar">
                <a href="/admin/">Dashboard</a>
                <a href="/admin/add-post.php">Add New Post</a>
                <a href="/admin/settings.php" class="active">Settings</a>
                <a href="/admin/logs.php">Logs</a>
            </aside>
            
            <div class="content">
                <h1>Settings</h1>
                
                <?php if ($error): ?>
                    <div style="background: #ef4444; color: white; padding: 1rem; border-radius: 6px; margin-bottom: 1rem;">
                        <?php echo $error; ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($message): ?>
                    <div style="background: #22c55e; color: white; padding: 1rem; border-radius: 6px; margin-bottom: 1rem;">
                        <?php echo $message; ?>
                    </div>
                <?php endif; ?>

                <form method="POST" enctype="multipart/form-data" style="display: flex; flex-direction: column; gap: 2rem;">
                    
                    <!-- Google Integration Section -->
                    <div class="post-card">
                        <h2 style="border-bottom: 1px solid var(--border); padding-bottom: 0.5rem; margin-bottom: 1rem;">Google Integration</h2>
                        
                        <div style="margin-bottom: 1.5rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Google Service Account JSON</label>
                            <p style="font-size: 0.875rem; color: var(--text-muted); margin-bottom: 0.5rem;">
                                Required for Google Indexing API (Search Console). Upload your <code>google-service-account.json</code> file here.
                            </p>
                            
                            <div style="display: flex; align-items: center; gap: 1rem;">
                                <input type="file" name="google_json" accept=".json" style="background: var(--bg-body); padding: 0.5rem; border-radius: 4px; border: 1px solid var(--border); color: var(--text-main);">
                                
                                <?php if ($googleJsonExists): ?>
                                    <span style="color: #22c55e; font-size: 0.875rem;">✓ File exists</span>
                                <?php else: ?>
                                    <span style="color: #eab308; font-size: 0.875rem;">⚠ File missing</span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div>
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Google Trends RSS Feeds</label>
                            <p style="font-size: 0.875rem; color: var(--text-muted); margin-bottom: 0.5rem;">
                                Comma-separated list of RSS feed URLs to scrape for trends.
                            </p>
                            <input type="text" name="rss_feeds" value="<?php echo htmlspecialchars($currentRss); ?>" style="width: 100%; padding: 0.75rem; background: var(--bg-body); border: 1px solid var(--border); color: white; border-radius: 6px;">
                        </div>
                    </div>

                    <!-- AI Configuration Section -->
                    <div class="post-card">
                        <h2 style="border-bottom: 1px solid var(--border); padding-bottom: 0.5rem; margin-bottom: 1rem;">AI Configuration</h2>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">AI API Key</label>
                                <input type="password" name="ai_api_key" value="<?php echo htmlspecialchars($currentAiKey); ?>" style="width: 100%; padding: 0.75rem; background: var(--bg-body); border: 1px solid var(--border); color: white; border-radius: 6px;">
                            </div>
                            
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">AI Model</label>
                                <input type="text" name="ai_model" value="<?php echo htmlspecialchars($currentAiModel); ?>" style="width: 100%; padding: 0.75rem; background: var(--bg-body); border: 1px solid var(--border); color: white; border-radius: 6px;">
                            </div>
                        </div>
                        
                        <div style="margin-top: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Plagiarism Threshold (0.0 - 1.0)</label>
                            <input type="number" step="0.1" min="0" max="1" name="plagiarism_threshold" value="<?php echo htmlspecialchars($currentThreshold); ?>" style="width: 100%; padding: 0.75rem; background: var(--bg-body); border: 1px solid var(--border); color: white; border-radius: 6px;">
                        </div>
                    </div>

                    <button type="submit" class="btn" style="padding: 1rem 2rem; font-size: 1.1rem;">Save Settings</button>
                </form>
            </div>
        </div>
    </main>
</body>
</html>
